"""
Tests for the implementation of feature sign search.
"""

__authors__ = "David Warde-Farley"
__copyright__ = "Copyright 2010-2011, Universite de Montreal"
__credits__ = ["David Warde-Farley"]
__license__ = "3-clause BSD"
__maintainer__ = "David Warde-Farley"
__email__ = "wardefar@iro"

import numpy as np
from pylearn2.optimization.feature_sign import feature_sign_search


class TestFeatureSign(object):
    @classmethod
    def setup_class(cls):
        rng = np.random.RandomState(0)
        cls.dictionary = rng.normal(size=(100, 500)) / 1000
        cls.dictionary /= np.sqrt((cls.dictionary ** 2).sum(axis=0))
        cls.gram = np.dot(cls.dictionary.T, cls.dictionary)
        cls.signal = rng.normal(size=100) / 1000
        cls.corr = np.dot(cls.dictionary.T, cls.signal)
        cls.nzi = [np.array([0, 6, 15, 30, 31, 34,
                             36, 48, 52, 53, 62, 94,
                             99, 103, 105, 107, 124, 131,
                             133, 137, 142, 145, 171, 178,
                             190, 206, 207, 214, 221, 226,
                             228, 229, 231, 234, 246, 255,
                             257, 258, 261, 265, 268, 269,
                             279, 281, 282, 288, 289, 293,
                             294, 296, 297, 302, 303, 306,
                             319, 322, 328, 329, 331, 335,
                             337, 351, 353, 355, 362, 368,
                             369, 372, 375, 380, 390, 392,
                             394, 397, 405, 410, 412, 420,
                             421, 422, 425, 430, 432, 435,
                             439, 443, 449, 451, 455, 456,
                             457, 458, 463, 479, 484, 485,
                             492, 493, 496]),
                   np.array([0, 6, 15, 30, 31, 34,
                             40, 52, 62, 84, 99, 103,
                             107, 131, 133, 137, 142, 145,
                             171, 190, 207, 221, 226, 228,
                             229, 231, 234, 243, 255, 257,
                             258, 261, 265, 268, 279, 281,
                             284, 288, 289, 293, 294, 296,
                             297, 302, 303, 306, 308, 319,
                             322, 329, 331, 335, 337, 351,
                             353, 355, 357, 362, 369, 371,
                             372, 375, 380, 390, 392, 394,
                             397, 405, 412, 420, 422, 430,
                             435, 439, 443, 451, 455, 456,
                             457, 463, 479, 483, 484, 485,
                             492, 493, 497]),
                   np.array([0, 6, 15, 21, 28, 31,
                             34, 40, 52, 62, 75, 103,
                             133, 164, 171, 190, 226, 228,
                             229, 231, 234, 255, 258, 261,
                             265, 268, 279, 284, 289, 293,
                             294, 296, 303, 306, 308, 319,
                             322, 335, 351, 355, 360, 369,
                             371, 372, 375, 380, 392, 394,
                             397, 420, 422, 430, 435, 439,
                             443, 451, 455, 456, 457, 463,
                             479, 483, 485, 492, 493, 497]),
                   np.array([0, 15, 21, 28, 31, 34,
                             40, 52, 62, 103, 164, 171,
                             190, 206, 226, 228, 229, 231,
                             234, 255, 258, 261, 265, 268,
                             279, 294, 303, 308, 319, 322,
                             335, 355, 360, 369, 371, 372,
                             375, 380, 392, 420, 422, 430,
                             439, 443, 446, 451, 456, 457,
                             479, 483, 485, 492, 497]),
                   np.array([0, 15, 31, 32, 34, 40,
                             52, 62, 103, 164, 190, 206,
                             226, 228, 229, 231, 234, 261,
                             268, 279, 294, 303, 308, 319,
                             322, 335, 355, 360, 363, 369,
                             371, 372, 375, 392, 422, 430,
                             439, 451, 457, 479, 485, 492, 497]),
                   np.array([0, 32, 40, 52, 103, 164,
                             182, 190, 229, 234, 268, 294,
                             303, 319, 335, 355, 360, 363,
                             369, 371, 372, 375, 392, 422,
                             430, 439, 479, 485, 492, 497]),
                   np.array([0, 32, 52, 103, 190, 229,
                             234, 268, 303, 319, 335, 363,
                             369, 371, 372, 375, 392, 422,
                             430, 439, 479]),
                   np.array([52, 190, 229, 234, 319, 335,
                             369, 371, 372, 375, 392, 422,
                             430, 479]),
                   np.array([52, 229, 234, 335, 369, 371,
                             372, 375, 422, 430, 479]),
                   np.array([229, 234, 369, 372, 375, 422, 479]),
                   np.array([234, 369, 375, 479]),
                   np.array([234, 369, 375, 479]),
                   np.array([234, 369]),
                   np.array([], dtype=np.int32)]
        cls.nze = [np.array([-1.9348249481460072e-03, -2.6733387284447153e-04,
                             -1.3132617311436409e-03, -4.8160077424667838e-04,
                             -7.5839249350565386e-04, -1.5909426774904003e-03,
                              5.7527969676995383e-05, 7.9447260531266320e-05,
                              1.0451809037644966e-03, -4.1613884795362038e-04,
                              1.5212190467793459e-03, 1.7787664944475309e-04,
                             -4.8836751622554329e-04, 1.6494321148399042e-03,
                             -2.7677800129602731e-06, -5.3707951770584682e-04,
                             -6.5444024170409589e-05, -1.5828515693151556e-04,
                             -7.2127417409642656e-04, -5.1401896787202958e-04,
                              3.7758489731919223e-05, 6.6680708523377810e-04,
                              5.2900822039974530e-04, 3.0975349526476766e-05,
                              1.2917361992321301e-03, -2.6589905870352928e-04,
                              4.5817183084630225e-05, 1.3754030475796234e-04,
                             -1.6320053069891164e-03, 4.5131596808813552e-04,
                             -2.5421398951701307e-04, 1.8433162492285078e-03,
                             -9.1173071359616081e-04, 1.7471082488497108e-03,
                             -5.3360585337509928e-06, 9.1283752277065869e-04,
                              2.6051969664101067e-05, -2.4057039933035281e-04,
                              4.1276967786097276e-04, -7.2185718488784083e-04,
                              7.3222222810349295e-04, -1.1258619069166660e-03,
                             -5.9539175662470837e-05, 1.2020202196530964e-04,
                             -6.4852933217228329e-05, -3.6136499613693474e-04,
                             -2.7509451425501420e-04, 7.8402482743400567e-04,
                             -1.6375047820679251e-03, -7.6328448464493817e-05,
                              2.5201048678107204e-04, -5.3343888345009510e-04,
                              1.3637930948889019e-03, 4.1072719924889650e-05,
                             -1.5727527265822143e-03, -1.5471429922949904e-03,
                             -5.6998443448416484e-04, -3.1120566643703262e-04,
                             -5.6386207057422957e-05, 5.7371833193473369e-04,
                              3.6188962138560557e-04, -4.7316772908796121e-04,
                              2.9484545185290238e-04, -6.6535357340597704e-04,
                              8.4119874640662382e-04, -2.8329017840323988e-05,
                             -2.5306839397483431e-03, 2.4715013439992202e-03,
                             -1.1910480261999676e-03, -2.9709479220933098e-04,
                              6.8431258887171015e-04, -2.7295761244718773e-03,
                             -2.4093971793902772e-05, 6.2201228574189945e-04,
                              5.1780698320989707e-04, 2.6733781310107836e-04,
                              5.6336177344038440e-04, -1.0616591760754300e-03,
                              3.6947848006075820e-04, 6.7141992520493947e-05,
                              2.9771817071157870e-04, 5.9797094277252722e-04,
                             -3.9933204819632510e-04, 8.6517244244875048e-04,
                              4.4920353680412715e-04, -6.6808772499122302e-04,
                             -5.3361944478563209e-04, 1.5242227367637563e-03,
                              5.2896850426062208e-04, 8.9662346856869842e-04,
                              5.5085221933474243e-04, 6.6736148020986167e-05,
                              8.1560784413259606e-04, 2.6011480255750508e-04,
                              6.6830208202212017e-04, 4.1085940128486484e-04,
                             -1.7682961625122970e-03, 8.3934219900091750e-04,
                             -2.0873017058309116e-05]),
                  np.array([-1.1035294455088696e-03, -2.2000890007574505e-04,
                            -7.8903889419539659e-04, -2.1119535900836997e-04,
                            -5.8279291450617342e-04, -1.7080583717419850e-03,
                            -2.6627718634094304e-04, 1.2487012342411459e-03,
                             1.2674860805574550e-03, 1.0316196854953281e-04,
                            -3.8989827573487992e-05, 1.3959573167727619e-03,
                            -2.9263690772758253e-04, -7.5516160142673238e-05,
                            -3.2027802224615773e-04, -4.0194333313972751e-05,
                             5.2792602866442842e-05, 1.0030885855889520e-04,
                             3.4359859807725127e-04, 1.4202578828080572e-03,
                             4.9435053415793168e-05, -2.9342239639478933e-04,
                             3.7342261630825908e-04, -1.1357718570444337e-04,
                             1.4693020881404192e-03, -8.1038826473469011e-04,
                             1.8665829767200055e-03, -9.6714718311873344e-05,
                             5.6412327489350621e-04, 7.0701526303205728e-05,
                            -3.9516191942322022e-04, 6.3432608784393975e-04,
                            -6.3954164541396520e-04, 4.5235975549458074e-04,
                            -8.5992159621133923e-05, 1.4804790922793453e-05,
                             4.0644969429524014e-05, -5.2722126671811232e-05,
                            -3.6077541732844342e-04, 4.2575453908280664e-04,
                            -1.1487697489090171e-03, -4.2018807922097055e-04,
                             3.4790809594425012e-05, -1.6160032947381991e-04,
                             1.2440480492768469e-03, 1.1231144781693225e-04,
                            -3.1889994757268988e-05, -1.4839438754208341e-03,
                            -1.4511548747212302e-03, -1.3590898496856349e-04,
                            -1.1193045928555190e-04, 1.0102752291013409e-03,
                             8.5488972321372484e-05, -4.5029116166219244e-04,
                             1.2167805925053460e-04, -9.8493755243442174e-04,
                             1.2491091198674258e-04, 3.7378349690722683e-04,
                            -2.7075691573085138e-03, -4.8129161730357662e-05,
                             2.6410719309027152e-03, -1.4938221269370467e-03,
                            -4.7855585389856826e-04, 1.6361688366860866e-04,
                            -2.1767953133433049e-03, -3.0577449369473172e-04,
                             1.9800957374812715e-04, 2.1713584774933896e-04,
                             1.2547286780562931e-04, -8.8978093970981957e-04,
                             1.8019125328437374e-04, 7.3691032485438852e-04,
                             6.7509982879728641e-04, 2.7311153047371414e-04,
                            -6.8500802556878382e-04, 1.3273193785611063e-03,
                             3.1869297588771220e-04, 8.6925548887159651e-04,
                             9.4018360395236385e-04, 3.0819494676812361e-04,
                             2.6271523986771711e-04, 3.0870274205584814e-04,
                             3.0289953214768245e-04, 1.3749791740379300e-04,
                            -1.1786735752273350e-03, 5.0250477725954619e-04,
                            -8.6070034229997804e-05]),
                  np.array([-8.3437579213689074e-04, -3.0015619460217073e-05,
                            -5.7897541676280300e-04, 1.6551471495779046e-05,
                            -2.2393605800632093e-04, -5.2001242860563247e-04,
                            -1.3870942715484462e-03, -5.9567578646616199e-04,
                             1.0534240091407204e-03, 9.1585112462395880e-04,
                            -7.4399294769562441e-05, 1.0651860600130367e-03,
                            -1.1115812798157547e-04, -1.2644694854842305e-04,
                             8.0859909014213599e-05, 9.5258814604793707e-04,
                             4.8694577190044154e-04, -1.2857867793752289e-04,
                             1.3623765533842772e-03, -6.3098386281712981e-04,
                             1.8709346040428150e-03, 2.4869691048019343e-04,
                            -4.4258285202598450e-04, 6.2641533196780173e-04,
                            -2.1070150676115288e-04, 5.5018802760393299e-04,
                            -8.6507017547800483e-05, 2.5113468709088080e-05,
                            -5.9572458159247166e-05, 2.7174990026237772e-04,
                            -6.8299151173242584e-04, -6.9141524100211209e-05,
                             9.9677709958487432e-04, 2.1455469032139378e-05,
                            -1.0931081545008502e-04, -1.4288398610195221e-03,
                            -9.2641619187052728e-04, 1.1631696831111123e-03,
                            -2.5008597010047157e-04, -4.9109092205391425e-04,
                             1.4573367169579149e-05, -2.4556619817285266e-03,
                            -2.2737442063480292e-04, 2.0873563680032670e-03,
                            -1.7221092085957831e-03, -3.9149748509506707e-04,
                            -1.9083890283104121e-03, -4.5179869240156825e-05,
                             3.1550562053295335e-08, -6.5469640615284299e-04,
                             2.3552832526085873e-04, 5.9969305766932981e-04,
                             2.2278964442494262e-04, 3.2359079954438926e-04,
                            -4.0264229418057637e-04, 7.7866637053198141e-04,
                             1.1841428769164845e-04, 3.8661136419945486e-04,
                             6.7602096499568042e-04, 3.1878480353931328e-04,
                             5.1411896426637957e-04, 2.5156200586072758e-04,
                             2.2088078507041245e-04, -1.1117185648776493e-03,
                             2.1497044788086782e-04, -1.0188107377535942e-04
                           ]),
                  np.array([-6.4713254630098830e-04, -2.7713153029343456e-04,
                             1.0757385692728924e-04, -1.2262501147393618e-04,
                            -2.3059082888739592e-04, -7.3161012508945306e-04,
                            -5.2413912135354621e-04, 6.2111326346937247e-04,
                             4.3335174642597019e-04, 8.9356967523135681e-04,
                            -3.3763242273813105e-04, 3.3430657447490773e-05,
                             6.2191414452732641e-04, -7.4132022612954130e-06,
                             5.3662340320133003e-04, -1.4178212558739250e-04,
                             1.3095988871201910e-03, -3.1597818931136630e-04,
                             2.0378913237038515e-03, 6.6562531139041254e-05,
                            -1.6868955601997661e-04, 2.8713705809760455e-04,
                            -1.2304693935549571e-04, 4.4141928313035041e-04,
                            -1.2253018622955899e-04, -4.6171051458478255e-04,
                             7.3723710785185957e-04, -1.9032537033656971e-04,
                            -1.2697562952863858e-03, -5.2482737659127472e-04,
                             1.2061124796596517e-03, -4.0817504185725420e-04,
                             2.5998882856128638e-05, -2.0966819569876341e-03,
                            -2.7608734302861750e-04, 1.6726478730511611e-03,
                            -1.6663463119142058e-03, -1.0586294396151170e-04,
                            -1.6941836184160486e-03, -2.5946770642156560e-04,
                             3.5633343009712243e-04, 6.9737388074824142e-04,
                             3.5050766641770261e-04, -6.2751827095089259e-05,
                             1.2404656899160754e-04, 3.5941380874020181e-04,
                             3.4446046190986252e-05, 4.2103096984376170e-04,
                             5.3026111408425991e-04, 5.1273187361432225e-05,
                             3.5432933330489038e-04, -7.4663640180765406e-04,
                            -3.8335424804281977e-04]),
                  np.array([-5.1836610661442424e-04, -1.0477410668018961e-04,
                            -5.7495834485112783e-05, 1.7900407122394070e-04,
                            -2.6703148129370935e-04, -3.9176848759322184e-04,
                             5.8091871319957496e-04, 1.6462922992537709e-04,
                             6.5002017969994672e-04, -3.2338128176255906e-04,
                             3.9479040414653807e-04, -2.7188461309840551e-05,
                             3.1135232326379012e-04, -2.8133206599649399e-05,
                             1.2641601901411529e-03, -1.0536469984737687e-04,
                             2.0205599857680634e-03, 1.0750521985466104e-05,
                             3.3091039446384973e-04, -6.4895294462220010e-05,
                            -2.1369409832986251e-04, 4.5576174900652586e-04,
                            -3.1012129723400486e-05, -1.1405083021739002e-03,
                            -8.5555280955904731e-05, 1.1637843624565714e-03,
                            -2.3051364523934005e-04, 3.4620317476115259e-05,
                             1.3934400265841262e-04, -1.6160179207995579e-03,
                            -3.5463876387457341e-04, 1.2476670438728542e-03,
                            -1.5272790887738484e-03, -1.2188452493555049e-03,
                             4.1391946602114946e-04, 5.7720967689920983e-04,
                             3.9160014684697103e-04, 7.9576153790110988e-05,
                             1.2980545083543301e-04, 5.7890412759707829e-04,
                             2.5677418172492382e-04, -3.4102744376433380e-04,
                            -3.8079926681336754e-04]),
                  np.array([-2.9393635354609123e-04, 2.8338417294713878e-04,
                            -7.1322391135519907e-05, 6.0928102145436805e-04,
                             4.1117051502882046e-04, -1.3597922568600674e-04,
                             3.5892355459974317e-06, 2.5647272365454443e-04,
                             1.0960456734638818e-03, 1.8670751666828435e-03,
                             1.7805877172321683e-04, -4.9436006980503034e-05,
                             2.8396916362562023e-04, -9.6659831830055534e-04,
                             9.8937911969777977e-04, -5.9732960363989054e-05,
                             6.0131277001765439e-06, 2.0633055937450479e-04,
                            -1.3238747581454256e-03, -4.4239264759708273e-04,
                             9.3309879025114137e-04, -1.3254818250397289e-03,
                            -7.4066565997860437e-04, 3.8820311348479301e-04,
                             4.3545224753910783e-04, 2.9649776286609744e-04,
                             6.5031628085758639e-04, 1.0771789715400958e-04,
                            -1.6633242528597823e-05, -1.2235122833187349e-04
                           ]),
                  np.array([-4.6467519143475876e-05, 1.0821123640127927e-04,
                             5.2011630846811980e-04, 1.5331854905217915e-04,
                             1.3866548495350116e-04, 8.2075898345678074e-04,
                             1.6343894091814725e-03, 1.0471892103365427e-04,
                             9.4287873260011381e-05, -6.0720057514918534e-04,
                             7.3927770528263870e-04, 7.3587103118350852e-05,
                            -1.2461689301653100e-03, -3.7588254684182553e-04,
                             7.0300087714795709e-04, -1.1703586963490017e-03,
                            -3.4761993323616724e-04, 3.2959907503653533e-04,
                             3.7357318166526934e-04, 1.2210075043889726e-04,
                             6.8778984271716630e-04]),
                  np.array([3.8272388058376711e-04, 1.9899166301244989e-05,
                             5.6189540105860936e-04, 1.3899178409636488e-03,
                            -2.5900269633353400e-04, 4.6538791234591143e-04,
                            -1.1071682569346324e-03, -2.6088142077321339e-04,
                             4.6133971870889563e-04, -9.8984072841656542e-04,
                            -3.7539187986192293e-06, 2.6845217976317101e-04,
                             2.4946606382563613e-04, 6.8309969710756292e-04
                           ]),
                  np.array([1.6472832434094874e-04, 3.4060230114089002e-04,
                            1.1427152271565732e-03, 1.9920279319872669e-04,
                            -9.6452219060174412e-04, -1.0812607675017613e-04,
                            2.7900553513436648e-04, -8.0628617478484157e-04,
                            1.5476496796440077e-04, 4.9104287902516419e-05,
                            6.1965472101718983e-04]),
                  np.array([1.7454378937657987e-04, 9.4360910010814669e-04,
                            -7.7576992971679156e-04, 8.8628243235214528e-05,
                            -6.1000343573593921e-04, 1.0084201597356237e-05,
                            5.3306901959571291e-04]),
                  np.array([0.000713547849065, -0.0005803708454116,
                            -0.0003480663897498, 0.000378159050198]),
                  np.array([4.6907206821826032e-04, -3.5326757239539523e-04,
                            -9.6124150391964280e-05, 1.9113349844253737e-04
                           ]),
                  np.array([0.000230238402802, -0.0001260715414588]),
                  np.array([], dtype=np.float64)]
        cls.penalties = [0.0001, 0.0006, 0.0011, 0.0016,
                         0.0021, 0.0026, 0.0031, 0.0036,
                         0.0041, 0.0046, 0.0051, 0.0056,
                         0.0061, 0.0066]

    @classmethod
    def teardown_class(cls):
        del cls.signal
        del cls.dictionary
        del cls.nzi
        del cls.nze
        del cls.penalties

    def test_driver(self):
        for index in range(len(self.penalties)):
            solution_vector = np.zeros(self.dictionary.shape[1])
            feature_sign_search(self.dictionary, self.signal,
                                self.penalties[index],
                                solution=solution_vector)
            yield self.check_against_reference, solution_vector, index
            yield self.check_zerocoef_optimality_cond, solution_vector, index
            yield self.check_nonzero_optimality_cond, solution_vector, index
            yield self.check_zeros_against_reference, solution_vector, index
            yield self.check_nonzeros_against_reference, solution_vector, index

    def check_zeros_against_reference(self, solution, index):
        z_ref = np.setdiff1d(np.arange(solution.shape[0]), self.nzi[index])
        z_ind = np.where(solution == 0)[0]
        assert z_ref.shape == z_ind.shape
        assert np.all(z_ref == z_ind)

    def check_nonzeros_against_reference(self, solution, index):
        nz_ref = self.nzi[index]
        nz_ind = np.where(solution != 0)[0]
        assert nz_ref.shape == nz_ind.shape
        assert np.all(nz_ref == nz_ind)

    def check_against_reference(self, solution, index):
        reference = np.zeros(self.dictionary.shape[1])
        reference[self.nzi[index]] = self.nze[index]
        assert np.allclose(solution, reference)

    def check_zerocoef_optimality_cond(self, solution, index):
        sparsity = self.penalties[index]
        grad = - 2 * self.corr + 2 * np.dot(self.gram, solution)
        grad[solution == 0]
        signs = np.sign(solution)
        assert np.all(abs(grad[signs == 0]) <= sparsity)

    def check_nonzero_optimality_cond(self, solution, index):
        sparsity = self.penalties[index]
        grad = - 2 * self.corr + 2 * np.dot(self.gram, solution)
        grad[solution == 0]
        signs = np.sign(solution)
        nzgrad = grad[signs != 0] + sparsity * signs[signs != 0]
        np.testing.assert_almost_equal(nzgrad, np.zeros(nzgrad.shape))

    def test_shape_rank_matches_1d_generated(self):
        sparsity = self.penalties[0]
        solution = feature_sign_search(self.dictionary, self.signal, sparsity)
        assert solution.ndim == 1
        assert solution.shape[0] == self.dictionary.shape[1]

    def test_shape_rank_matches_2d_generated(self):
        sparsity = self.penalties[0]
        signal = self.signal.reshape(1, -1)
        solution = feature_sign_search(self.dictionary, signal, sparsity)
        assert solution.ndim == 2
        assert solution.shape[0] == 1
        assert solution.shape[1] == self.dictionary.shape[1]

    def test_solution_identity_1d_provided(self):
        sparsity = self.penalties[0]
        solution = np.zeros(self.dictionary.shape[1])
        newsol = feature_sign_search(self.dictionary, self.signal, sparsity,
                                     solution=solution)
        assert solution is newsol

    def test_solution_identity_2d_provided(self):
        sparsity = self.penalties[0]
        solution = np.zeros((1, self.dictionary.shape[1]))
        signal = self.signal.reshape(1, -1)
        newsol = feature_sign_search(self.dictionary, signal, sparsity,
                                     solution=solution)
        assert solution is newsol
